#!/opt/anaconda3/bin/python
import sys,math,subprocess,termios
from PIL import Image # this is for Pillow
# You need to install pillow by pip or conda

# ph2gif.py
# made by mkanzaki@me.com
# created:  2023 02/04: first version from ph2movie.py: 3 steps to make gif movie
# modified: 2023 02/05: imported vibration_movie.py, but still two steps
# modified: 2023 02/06: now use Pillow to make gif: single step to gif movie
# modified: 2023 02/06: and renamed as ph2gif.py
# modified: 2023 10/13: some improvements
# modified: 2023 11/15: simplified to make this code public  

# This code reads the output file of Quantum-Espresso's ph.x program, etc. 
# and exports a gif movie file of the displaced crystal structure according
# to the displacement of the selected vibrational mode. 

# Three files, *.out (ph.x output), *.dyn and *.vesta are necessary
# Last one, VESTA file should be made before using this code.
# You can make a VESTA file using pw2xtl.py and an output file from pw.x. 
# Then import it (xtl) from VESTA, and change orientation, magnification, 
# bond, polyhedra etc. which you want to see for final gif movie, 
# and then save it as a VESTA file.

# Note:
# 1) structural coordinates and displacements are obtained from ph.output file, and 
#    all others are from the VESTA file.
# 2) During the process, many files are produced, but they will be deleted at last.
# 3) You can rotate or enlarge/shrink the crystal during the vibrational movements, 
#    to do so, change line near 417 for rotation for example.
# 4) You better use shorter and larger bond distance min and max values than normal, 
#     as bond distance will change largely during the vibrational motion.

# Usage:
# > ./ph2gif.py ph-out-file dyn-file vesta-file
# first argument: phonon output file name
# second argument: *.dyn file name
# third argument: *.vesta file name
#
# A list of vibrational modes will be displayed.
# Select mode by number, then the corresponding gif file will be produced.
# It will take while, be patient.
# Type q to quit.
# Type 0 to change setting, such as displacement magnitude.

fd = sys.stdin.fileno() # to get key input
old = termios.tcgetattr(fd)
new = termios.tcgetattr(fd)
new[3] &= ~termios.ICANON
new[3] &= ~termios.ECHO

celldm = [0.0 for i in range(7)]
mat1 = [0.0 for i in range(4)]
mat2 = [0.0 for i in range(4)]
mat3 = [0.0 for i in range(4)]
rmat1 = [0.0 for i in range(4)]
rmat2 = [0.0 for i in range(4)]
rmat3 = [0.0 for i in range(4)]
atom = ['' for i in range(500)]
freq = [0.0 for i in range(500)]
mode = ['' for i in range(500)]
x = [0.0 for i in range(500)]
y = [0.0 for i in range(500)]
z = [0.0 for i in range(500)]
dx = [[0.0 for i in range(500)] for j in range(500)]
dy = [[0.0 for i in range(500)] for j in range(500)]
dz = [[0.0 for i in range(500)] for j in range(500)]

if len(sys.argv) < 4:
	print('Not enough arguments provided!')
	print('For example: > ./ph2gif.py ph_out_file dyn_file vesta_file')
	print('First argument: ph.out file name')
	print('Second argument: .dyn file name')
	print('Third argument: vesta file name')
	print('Read header of this code for details.')
	exit()

scale = 1.0 # how much atoms are displaced, change if necessary
nstep = 21 # number of states to be prepared, odd number is better, larger for fine movement

# reading argument
base = sys.argv[1]

# change below according to your preferred file naming convention 
f1 = sys.argv[1]  # for reading crystal structure and mode info
f2 = sys.argv[2] # for reading vibrational displacements
f5 = sys.argv[3] # for reading vesta file
try:
	file1 = open(f1,'r')
except IOError as xxx_todo_changeme:
	(errno, msg) = xxx_todo_changeme.args
	print(f1 + ' file open error!')
	exit()
try:
	file2 = open(f2,'r')
except IOError as xxx_todo_changeme1:
	(errno, msg) = xxx_todo_changeme1.args
	print(f2 + ' file open error!')
	exit()
try:
	file5 = open(f5,'r')
except IOError as xxx_todo_changeme1:
	(errno, msg) = xxx_todo_changeme1.args
	print(f5 + ' file open error!')
	exit()

# define constants
rd = math.pi/180.0
bohr = 0.5291772108 # used to convert from bohr to Ang.

all_lines1 = file1.readlines() # read ph.out 
file1.close()
findex = 0 # position of current file1 or file2 lines

# Get Bravais index (cell) from *.ph.out
for i in range(0,len(all_lines1)): # find bravais-lattice index line
	ftext = all_lines1[i]
	if 'bravais-lattice index' in ftext :
	   break
s =ftext.split()
ibrav = int(s[3]) # ibrav number integer
findex = i # last i

# Get alat from *.out
ftext = all_lines1[findex + 1]
s =ftext.split()
alat = float(s[4]) # alat (celldm[1])
#print(alat)

# Number of atoms in cell
for i in range(0,len(all_lines1)): # find bravais-lattice index line
	ftext = all_lines1[i]
	if 'number of atoms/cell' in ftext :
	   break
s =ftext.split()
natom = int(s[4]) # number of atoms in a cell
nmode = natom * 3 # xyz freedom times total number of atoms in the unit cell
findex = i # last i

#read crystal axes (necessary to convert to cell parameters)
for i in range(findex,len(all_lines1)): # find crystal axes: line
	ftext = all_lines1[i]
	if 'crystal axes:' in ftext :
		break
findex = i
ftext = all_lines1[findex+1] # next line
s = ftext.split()
mat1[1] = float(s[3])
mat1[2] = float(s[4])
mat1[3] = float(s[5])
ftext = all_lines1[findex+2] # next line
s = ftext.split()
mat2[1] = float(s[3])
mat2[2] = float(s[4])
mat2[3] = float(s[5])
ftext = all_lines1[findex+3] # next line
s = ftext.split()
mat3[1] = float(s[3])
mat3[2] = float(s[4])
mat3[3] = float(s[5])
findex = findex + 4

#read reciprocal axes (necessary for converting coordinates)
for i in range(findex,len(all_lines1)): # find crystal axes: line
	ftext = all_lines1[i]
	if 'reciprocal axes:' in ftext :
		break
findex = i
ftext = all_lines1[findex+1] # next line
s = ftext.split()
rmat1[1] = float(s[3])
rmat1[2] = float(s[4])
rmat1[3] = float(s[5])
ftext = all_lines1[findex+2] # next line
s = ftext.split()
rmat2[1] = float(s[3])
rmat2[2] = float(s[4])
rmat2[3] = float(s[5])
ftext = all_lines1[findex+3] # next line
s = ftext.split()
rmat3[1] = float(s[3])
rmat3[2] = float(s[4])
rmat3[3] = float(s[5])
findex = findex + 4
# print(rmat1[1],rmat1[2],rmat1[3],rmat2[1],rmat2[2],rmat2[3],rmat3[1],rmat3[2],rmat3[3])

# calculate cell parameters from alat, mat matrix
cella = math.sqrt(mat1[1]*mat1[1]+mat1[2]*mat1[2]+mat1[3]*mat1[3])
cellb = math.sqrt(mat2[1]*mat2[1]+mat2[2]*mat2[2]+mat2[3]*mat2[3])
cellc = math.sqrt(mat3[1]*mat3[1]+mat3[2]*mat3[2]+mat3[3]*mat3[3])
alpha = math.acos((mat2[1]*mat3[1]+mat2[2]*mat3[2]+mat2[3]*mat3[3])/(cellb*cellc))/rd
beta = math.acos((mat1[1]*mat3[1]+mat1[2]*mat3[2]+mat1[3]*mat3[3])/(cella*cellc))/rd
gamma = math.acos((mat1[1]*mat2[1]+mat1[2]*mat2[2]+mat1[3]*mat2[3])/(cella*cellb))/rd
cella = alat*bohr*cella
cellb = alat*bohr*cellb
cellc = alat*bohr*cellc
#print(cella,cellb,cellc,alpha,beta,gamma)

# read atomic coordinates (they are alat units, so conversion is necessary) 
# For conversion, reciprocal axes rmat[] are used 

for i in range(findex,len(all_lines1)): # find site n. atom... line
	ftext = all_lines1[i]
	if 'site n.  atom' in ftext :
		break
findex = i + 1 # point next line
for i in range(0,natom): # find site n. atom... line
	ftext = all_lines1[findex+i]
	s = ftext.split()
	atom[i+1] = s[1]
	x1 = float(s[7]) # ph output need +1 compared to scf/relax out as mass[] is included
	y1 = float(s[8])
	z1 = float(s[9])
	x[i+1]= x1*rmat1[1] + y1*rmat1[2] + z1*rmat1[3] # conversion
	y[i+1]= x1*rmat2[1] + y1*rmat2[2] + z1*rmat2[3] # conversion 
	z[i+1]= x1*rmat3[1] + y1*rmat3[2] + z1*rmat3[3] # conversion
#	print(x[i+1],y[i+1],z[i+1])
findex = findex + natom # point current line

# find Mode symmetry part
for i in range(findex,len(all_lines1)): # find total number of modes
	ftext = all_lines1[i]
	if '***************' in ftext :
		break
ij = i
for i in range(ij+1,len(all_lines1)): # find total number of modes
	ftext = all_lines1[i]
	if '***************' in ftext :
		break
findex = i + 1 # last i

# find line contains Mode symmetry
for i in range(findex,len(all_lines1)): # find site n. atom... line
	ftext = all_lines1[i]
	if 'Mode symmetry,' in ftext :
		#print(ftext) # print Mode symmetry,...
		break
findex = i + 2 # point next^2 line

# obtain number of actual lines for mode symmetry 
# Not equal to nmode if this contains E or T modes and has cneter of symmetry
m = 0
for i in range(findex,findex + nmode): # find site n. atom... line
	ftext = all_lines1[i]
	if 'freq' in ftext :
		m = m + 1
	else:
		break
nmode2 = m
if nmode != nmode2:
	print('Degenerated modes...')

# read character representation (B_1g etc.) and vibrational frequency
imode = 1
for i in range(0,nmode2): # read frequency and mode 
	ftext = all_lines1[findex+i]
	s = ftext.split()
	#print(s)
	s1 = s[2]
	if s1[len(s1)-1] == '-': # 6.7 or later QE ph output need this
		s1 = s1.rstrip('-')
		m1 = int(s1)
		f1 = float(s[5])
		str1 = s[8]
		s2 = s[3]
		s2 = s2.rstrip(')')
		m2 = int(s2)	
	else: # for older QE ph output
		m1 = int(s1)
		f1 = float(s[6])
		str1 = s[9]
		s2 = s[4]
		s2 = s2.rstrip(')')
		m2 = int(s2)	
	j = m2 - m1 + 1
	for k in range(1,j+1):
		mode[imode] = str1
		freq[imode] = f1
		imode = imode + 1
imode = imode -1
#print(nmode)

# read from dyn (file2)
all_lines2 = file2.readlines()
file2.close()
findex = 1
for i in range(findex,len(all_lines2)): # find Diagonalizing the dynamical matrix
	ftext = all_lines2[i]
	if 'Diagonalizing the dynamical matrix' in ftext :
		break
findex = i # last i
for i in range(findex,len(all_lines2)): # Then find ****** line
	ftext = all_lines2[i]
	if '******' in ftext :
		break
findex = i # last i
if (findex + 1) >= (len(all_lines2)): # if something wrong...
   print('Unexpectedly end of line reached during reading dyn file!')
   exit(0)
# read displacement vectors dx[mode][atom],dy[mode][atom],dz[mode][atom]
for i in range(1,imode+1): 
	findex = findex + 1
	for j in range(1,natom+1):
		ftext = all_lines2[findex + j]
		s = ftext.split()
		dx[i][j] = float(s[1]) # just read raw values
		dy[i][j] = float(s[3])
		dz[i][j] = float(s[5])
	findex = findex + natom

# read from VESTA file (file5)
all_lines5 = file5.readlines()
file5.close()

# Start VESTA
cmd = 'open -a /Applications/VESTA/VESTA.app' + ' ' +  f5
#cmd = "open -a /Applications/VESTA/VESTA.app" 
subprocess.call(cmd,shell=True)
cmd = "sleep 1.0"  # wait a while 
subprocess.call(cmd,shell=True)
f6 = 'temp.vstx'
angle = 0.0
# nstep defined at line 76, and default nstep = 21
# Make list to arrange order to be shown in a movie...

# Interactively select mode, and make a VESTA file for the selected mode
while True: # print list of the modes
	for i in range(1,imode+1):
		print(' ' + str(i) + '   ' + mode[i] + ' ' + str(freq[i]))
	print('Select mode by number (0 for change setting: q for quit)') # prompt for input
	stop = False
	tmp = input('>>> ')
	if tmp.isalpha():
		if tmp == 'q' or tmp == 'Q' :
			stop = True
			qmode = 100000
	else:
		qmode = int(tmp)
	if stop: # if q, exit
		exit()
	if qmode == 0: # if 0, change style
		while True:
			print('Select number') # prompt for input
			print('1: change displacement magnitude? current = ' + str(scale))
			print('2: number of steps? current = ' + str(nstep))
			i = int(input('>>> '))
			if i > 0 and i< 3:
				if i == 1:
					print('Input new displacement magnitude')
					scale = float(input('>>> '))
					break
				elif i == 2:
					print('Input number of steps')
					nstep = int(input('>>> '))
					break
	if qmode > 0 and qmode < imode+1: # if given mode no. is OK, then proceed
		# Output for VESTA file loop
		nn = nstep // 2
		#nnn = nstep // 4 # 
		base2 = base + '-' + mode[qmode].strip() + '-' + str('%04.1f' % freq[qmode]) + 'cm-1'
		for j in range(0,nstep-1): # make VESTA files
			f3 = base2 + '_' + str(j) + '.vesta' # file name + mode name etc.
			file3 = open(f3,'w')
			file3.write('#VESTA_FORMAT_VERSION 3.5.4\n') # need to change later?
			file3.write('CRYSTAL\n')
			file3.write('TITLE\n')
			file3.write('ph.x: ' + base2 + '\n') # comment
			for i in range(0,len(all_lines5)):
				ftext = all_lines5[i]
				if 'TITLE' in ftext:
					ix = i + 2
					break
			for i in range(ix,len(all_lines5)):
				ftext = all_lines5[i]
				if 'STRUC' in ftext:
					ix = i
					break
				file3.write(ftext)
			file3.write('STRUC\n')
			m = 0.001*(nn*nn - (j - nn)*(j - nn)) # add speed: adjust the factor
			if (j - nn) < 0:
				factor = -m*scale 
			else:
				factor = m*scale
			for k in range(1,natom+1): # atomic positions with displacements
				nx = x[k] + factor*dx[qmode][k]  
				ny = y[k] + factor*dy[qmode][k]
				nz = z[k] + factor*dz[qmode][k]
				file3.write('  ' + str(k) + ' ' + atom[k] + '         ' + atom[k] + ' 1.0000 ' + str('%02.6f' % nx) + ' ' + str('%02.6f' % ny) + ' ' + str('%02.6f' % nz) + '    1a       1\n')
				file3.write('                            0.000000   0.000000   0.000000  0.00\n')	
			file3.write('  0 0 0 0 0 0 0\n') 
			for i in range(0,len(all_lines5)):
				ftext = all_lines5[i]
				if 'THERI' in ftext:
					ix = i
					break
			for i in range(ix,len(all_lines5)):
				file3.write(all_lines5[i])
			file3.close()
		# VESTA files with displaced coordinates are generated.
		list = [] 
		picture = [] # for Pillow
		for h in range(0,nn): # arrange order of pictures to be shown in movie
			list.append(h)
		for h in reversed(range(0,nn-1)):
			list.append(h)
		for h in reversed(range(nn+1,nstep-1)):
			list.append(h)
		for h in range(nn+1,nstep-1):
			list.append(h)
		# produce png files and make gif file
		ix = 0
		for i in list: # 
			s0 = '-open' + ' ' + base2 + '_' + str(i) + '.vesta\n'
			# To rotate the crystal, uncomment following two lines and line 429 
			#angle = angle + 9.23 # for rotation
			#s1 = '-rotate_y ' + str(angle) + '\n' # rotation around y-axis
			#
			# export image as png
			s3 = '-export_img ' + base2 + '_' + str(ix) + '.png\n'
			file6 = open(f6,'w')
			if i > 0:
				s2 = '-close\n' # to close vesta file. uncomment this line if you want to keep them
				file6.write(s2)
			file6.write(s0)
			# for rotation, uncomment below
			#file1.write(s1)
			file6.write(s3)
			file6.close()
			cmd = 'open -a /Applications/VESTA/VESTA.app' + ' ' +  f6 
			subprocess.call(cmd,shell=True)
			cmd = 'sleep 1.0'   # for timing, may be changed, but small number such as 0.1 may cause error
			subprocess.call(cmd,shell=True)
			png_name = base2 + '_' + str(ix) + '.png' # Pillow
			img = Image.open(png_name) # Pillow
			picture.append(img) # Pillow
			picture[0].save(base2+'.gif',save_all=True, append_images=picture[1:], optimize=True, duration=50, loop=0) # duration may be changed
			cmd = 'rm' + ' ' + png_name # remove png   If you want to keep png files, uncomment this and next line
			subprocess.call(cmd,shell=True)
			ix = ix + 1
		file6.close()
		cmd = 'rm' + ' ' + f6   # remove temp file
		subprocess.call(cmd,shell=True)
		for k in range(0,nstep): # Remove vesta file. If you want to keep files, uncommnet this and next two lines
			cmd = 'rm' + ' ' + base2 + '_' + str(k) + '.vesta'  
			subprocess.call(cmd,shell=True)

exit()
